import 'package:admin_desktop/app_constants.dart';
import 'package:admin_desktop/application/profile/profile_provider.dart';
import 'package:admin_desktop/application/right_side/right_side_provider.dart';
import 'package:admin_desktop/application/select_address/select_address_provider.dart';
import 'package:admin_desktop/domain/models/data/address_data.dart';
import 'package:admin_desktop/domain/models/data/location_data.dart';
import 'package:admin_desktop/infrastructure/services/app_helpers.dart';
import 'package:admin_desktop/infrastructure/services/local_storage.dart';
import 'package:admin_desktop/infrastructure/services/tr_keys.dart';
import 'package:admin_desktop/presentation/app_assets.dart';
import 'package:admin_desktop/presentation/components/buttons/custom_mini_button.dart';
import 'package:admin_desktop/presentation/components/buttons/pop_button.dart';
import 'package:admin_desktop/presentation/components/login_button.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/right_side/address/search_form_field.dart';
import 'package:admin_desktop/presentation/pages/main/widgets/right_side/address/tile_providers.dart';
import 'package:admin_desktop/presentation/styles/style.dart';
import 'package:auto_route/auto_route.dart';
import 'package:flutter/material.dart';
import 'package:flutter_map/flutter_map.dart';
import 'package:flutter_remix/flutter_remix.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:flutter_screenutil/flutter_screenutil.dart';

// ignore: depend_on_referenced_packages
import 'package:latlong2/latlong.dart';
import 'package:lottie/lottie.dart' as lottie;

class FlutterMapPage extends ConsumerStatefulWidget {
  final ValueChanged<AddressData> onSelect;
  final LocationData? location;
  final bool? isShopLocation;

  const FlutterMapPage({
    super.key,
    required this.onSelect,
    this.location,
    this.isShopLocation,
  });

  @override
  ConsumerState<FlutterMapPage> createState() => _FlutterMapPageState();
}

class _FlutterMapPageState extends ConsumerState<FlutterMapPage>
    with TickerProviderStateMixin {
  late AnimationController _animationController;
  late MapController mapController;

  static const _startedId = 'AnimatedMapController#MoveStarted';
  static const _inProgressId = 'AnimatedMapController#MoveInProgress';
  static const _finishedId = 'AnimatedMapController#MoveFinished';

  @override
  void initState() {
    WidgetsBinding.instance.addPostFrameCallback((s) {
      ref.read(selectAddressProvider.notifier).fetchLocationName(LocationData(
            longitude: (widget.isShopLocation ?? false)
                ? widget.location?.longitude
                : AppHelpers.getInitialLongitude(),
            latitude: (widget.isShopLocation ?? false)
                ? widget.location?.latitude
                : AppHelpers.getInitialLatitude(),
          ));
    });
    super.initState();

    mapController = MapController();
    _animationController = AnimationController(vsync: this);
    // showIntroDialogIfNeeded();
  }

  @override
  void dispose() {
    super.dispose();
    _animationController.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final state = ref.watch(selectAddressProvider);
    final event = ref.read(selectAddressProvider.notifier);
    return SizedBox(
        width: MediaQuery.sizeOf(context).width,
        //height: 300,
        child: Stack(
          children: [
            ClipRRect(
              borderRadius: BorderRadius.circular(12.r),
              child: FlutterMap(
                mapController: mapController,
                options: MapOptions(
                  maxZoom: 20,
                  minZoom: 3,
                  initialCenter: LatLng(
                    widget.location?.latitude ??
                        AppHelpers.getInitialLatitude(),
                    widget.location?.longitude ??
                        AppHelpers.getInitialLongitude(),
                  ),
                  initialZoom: 16,
                  onPointerDown: (d, f) {
                    _animationController.repeat(
                      min: AppConstants.pinLoadingMin,
                      max: AppConstants.pinLoadingMax,
                      period: _animationController.duration! *
                          (AppConstants.pinLoadingMax -
                              AppConstants.pinLoadingMin),
                    );
                    event.setChoosing(true);
                  },
                  onPointerUp: (d, f) {
                    event.fetchLocationName(LocationData(
                        latitude: mapController.camera.center.latitude,
                        longitude: mapController.camera.center.longitude));

                    _animationController.forward(
                      from: AppConstants.pinLoadingMax,
                    );
                    event.setChoosing(false);
                  },
                ),
                children: [
                  openStreetMapTileLayer,
                ],
              ),
            ),
            IgnorePointer(
              child: Center(
                child: Padding(
                  padding: const EdgeInsets.only(
                    bottom: 78.0,
                  ),
                  child: lottie.Lottie.asset(
                    Assets.lottiePin,
                    onLoaded: (composition) {
                      _animationController.duration = composition.duration;
                    },
                    controller: _animationController,
                    width: 250,
                    height: 250,
                  ),
                ),
              ),
            ),
            Column(
              mainAxisSize: MainAxisSize.min,
              children: [
                54.verticalSpace,
                Container(
                  height: 50.r,
                  padding: REdgeInsets.symmetric(horizontal: 16),
                  margin: REdgeInsets.symmetric(horizontal: 16),
                  decoration: BoxDecoration(
                    boxShadow: const <BoxShadow>[
                      BoxShadow(
                        color: Style.greyColor,
                        offset: Offset(0, 2),
                        blurRadius: 2,
                        spreadRadius: 0,
                      ),
                    ],
                    color: Style.white,
                    borderRadius: BorderRadius.circular(25.r),
                  ),
                  child: Row(
                    children: [
                      Icon(
                        FlutterRemix.search_line,
                        size: 20.r,
                        color: Style.black,
                      ),
                      12.horizontalSpace,
                      SearchFormField(
                          controller: state.textController,
                          onChanged: (d) {
                            event.setQuery(context);
                          }),
                      IconButton(
                        onPressed: event.clearSearchField,
                        splashRadius: 20.r,
                        padding: EdgeInsets.zero,
                        icon: Icon(
                          FlutterRemix.close_line,
                          size: 20.r,
                          color: Style.black,
                        ),
                      ),
                    ],
                  ),
                ),
                if (state.isSearching || state.searchedPlaces.isNotEmpty)
                  Container(
                    decoration: BoxDecoration(
                      borderRadius: BorderRadius.circular(15.r),
                      color: Style.white,
                    ),
                    margin: REdgeInsets.symmetric(horizontal: 16, vertical: 6),
                    padding:
                        REdgeInsets.symmetric(horizontal: 20, vertical: 10),
                    child: state.isSearching
                        ? _customLoading()
                        : ListView.builder(
                            shrinkWrap: true,
                            itemCount: state.searchedPlaces.length,
                            padding: EdgeInsets.only(bottom: 22.h),
                            itemBuilder: (context, index) {
                              return InkWell(
                                onTap: () {
                                  _animatedMapMove(
                                      LatLng(state.searchedPlaces[index].lat,
                                          state.searchedPlaces[index].lon),
                                      15);
                                  event.goLocationOpen(
                                      place: state.searchedPlaces[index]);
                                },
                                child: Column(
                                  crossAxisAlignment: CrossAxisAlignment.start,
                                  children: [
                                    22.verticalSpace,
                                    Text(
                                      state.searchedPlaces[index]
                                              .address?["country"] ??
                                          "",
                                    ),
                                    Text(
                                      state.searchedPlaces[index].displayName,
                                      maxLines: 2,
                                      overflow: TextOverflow.ellipsis,
                                    ),
                                    const Divider(color: Style.greyColor),
                                  ],
                                ),
                              );
                            }),
                  ),
              ],
            ),
            state.isLoading
                ? SizedBox.shrink()
                : Positioned(
                    bottom: 120.r,
                    right: 15.r,
                    child: Column(
                      children: [
                        CustomMiniButton(
                            onTap: () {
                              mapController.move(mapController.camera.center,
                                  mapController.camera.zoom + 1);
                            },
                            icon: Icons.add),
                        5.verticalSpace,
                        CustomMiniButton(
                            onTap: () {
                              mapController.move(mapController.camera.center,
                                  mapController.camera.zoom - 1);
                            },
                            icon: Icons.remove),
                      ],
                    ),
                  ),
            AnimatedPositioned(
              duration: const Duration(milliseconds: 150),
              bottom: state.isChoosing ? -60.r : 32.r,
              left: 15.r,
              right: 15.r,
              child: Row(
                mainAxisAlignment: MainAxisAlignment.spaceBetween,
                children: [
                  PopButton(
                    popSuccess: () {
                      if (widget.isShopLocation ?? false) {
                        ref.read(profileProvider.notifier).changeIndex(1);
                      } else {
                        context.maybePop();
                      }
                    },
                  ),
                  const Spacer(),
                  CustomMiniButton(
                    onTap: () {
                      event.goToMyLocation((loc) {
                        _animatedMapMove(
                            LatLng(loc.latitude ?? 0, loc.longitude ?? 0), 15);
                      });
                    },
                    icon: !state.searchingMyLocation
                        ? FlutterRemix.navigation_line
                        : null,
                    size: 56,
                    child: state.searchingMyLocation
                        ? Padding(
                            padding: const EdgeInsets.all(12),
                            child: CircularProgressIndicator(
                              color: Style.black,
                            ),
                          )
                        : null,
                  ),
                  16.horizontalSpace,
                  SizedBox(
                    width: 200.w,
                    child: LoginButton(
                        height: 56,
                        isLoading: state.isButtonLoading,
                        title:
                            AppHelpers.getTranslation(TrKeys.confirmLocation),
                        onPressed: () {
                          final LocationData location = LocationData(
                            longitude: mapController.camera.center.longitude,
                            latitude: mapController.camera.center.latitude,
                          );

                          final AddressData addressData = AddressData(
                            location: location,
                            zipCode: state.zipCode,
                            address: state.textController?.text ?? "",
                          );

                          if (widget.isShopLocation ?? false) {
                            if (state.textController?.text.isEmpty ?? true) {
                              ref
                                  .read(selectAddressProvider.notifier)
                                  .fetchLocationName(
                                    location,
                                  )
                                  .then((d) {
                                addressData;
                                ref
                                    .read(profileProvider.notifier)
                                    .changeIndex(1);
                              });
                            } else {
                              widget.onSelect(addressData);
                              ref.read(profileProvider.notifier).changeIndex(1);
                            }
                          } else {
                            final rightSideState = ref.watch(rightSideProvider);
                            final AddressData? address = LocalStorage.getBags()[
                                    rightSideState.selectedBagIndex]
                                .selectedAddress;
                            if (!(state.textController?.text
                                    .contains(address?.title ?? '') ??
                                false)) {
                              AppHelpers.showSnackBar(
                                  context,
                                  AppHelpers.getTranslation(
                                      TrKeys.thisLocationIsWrong));
                              return;
                            } else {
                              if (state.textController?.text.isEmpty ?? true) {
                                ref
                                    .read(selectAddressProvider.notifier)
                                    .fetchLocationName(
                                      location,
                                    )
                                    .then((d) {
                                  ref
                                      .read(profileProvider.notifier)
                                      .changeIndex(1);
                                  widget.onSelect(
                                    addressData,
                                  );

                                  if (context.mounted) {
                                    context.maybePop();
                                  }
                                });
                              } else {
                                widget.onSelect(
                                  addressData,
                                );
                                context.maybePop();
                              }
                            }
                          }
                        }),
                  ),
                ],
              ),
            ),
          ],
        ));
  }

  void _animatedMapMove(LatLng destLocation, double destZoom) {
    final camera = mapController.camera;
    final latTween = Tween<double>(
        begin: camera.center.latitude, end: destLocation.latitude);
    final lngTween = Tween<double>(
        begin: camera.center.longitude, end: destLocation.longitude);
    final zoomTween = Tween<double>(begin: camera.zoom, end: destZoom);

    final controller = AnimationController(
        duration: const Duration(milliseconds: 500), vsync: this);
    final Animation<double> animation =
        CurvedAnimation(parent: controller, curve: Curves.fastOutSlowIn);
    final startIdWithTarget =
        '$_startedId#${destLocation.latitude},${destLocation.longitude},$destZoom';
    bool hasTriggeredMove = false;

    controller.addListener(() {
      final String id;
      if (animation.value == 1.0) {
        id = _finishedId;
      } else if (!hasTriggeredMove) {
        id = startIdWithTarget;
      } else {
        id = _inProgressId;
      }

      hasTriggeredMove |= mapController.move(
        LatLng(latTween.evaluate(animation), lngTween.evaluate(animation)),
        zoomTween.evaluate(animation),
        id: id,
      );
    });

    animation.addStatusListener((status) {
      if (status == AnimationStatus.completed) {
        controller.dispose();
      } else if (status == AnimationStatus.dismissed) {
        controller.dispose();
      }
    });

    controller.forward();
  }

  Widget _customLoading() {
    return Container(
        height: 50,
        width: MediaQuery.sizeOf(context).width,
        decoration: BoxDecoration(),
        child: Text("Searching..."));
  }
}
